import javafx.scene.*;
import javafx.scene.paint.Color;
import javafx.scene.paint.PhongMaterial;
import javafx.scene.shape.Sphere;
import javafx.scene.transform.Rotate;
import javafx.scene.image.Image;

/**
 * SunBody: 太陽系の中心で自ら輝く恒星クラス。
 * * 【設計思想】
 * 他の惑星と異なり公転はせず、原点(0,0,0)に固定される。
 * 視覚的には、外部のライト設定に依存せず、常に指定のテクスチャや色が
 * 最大輝度で表示される「自己発光(Self-Illumination)」設定を適用し、
 * 宇宙の光源としての存在感を演出する。
 * @author きん ＆ Gemini
 */
public class SunBody extends CelestialBody {

	private double radius;		  // 太陽の半径
	private String texturePath;	// 表面テクスチャのパス
	private Color sunColor;		 // テクスチャ未ロード時の代替色

	private double rotationAngle = 0;   // 自転角
	private double rotationSpeed = -0.5; // 自転速度（負の値で時計回り等の調整）

	/**
	 * コンストラクタ
	 * @param nameJa	  天体名（日）
	 * @param nameEn	  天体名（英）
	 * @param radius	  太陽の半径
	 * @param texturePath 太陽表面のテクスチャパス
	 * @param sunColor	テクスチャがない場合の発光色
	 */
	public SunBody(String nameJa, String nameEn, double radius, String texturePath, Color sunColor) {

		// 太陽は公転しないため、基底クラスの公転スピードは 0
		super(nameJa, nameEn, 0);

		this.radius = radius;
		this.texturePath = texturePath;
		this.sunColor = sunColor;

		// 初期座標は太陽系の中心（原点）に固定
		this.x = 0;
		this.y = 0;
		this.z = 0;

		// 3Dモデルと質感の構築
		this.setupAppearance();

	} // end SunBody constructor

	/**
	 * setupAppearance: 太陽特有の「光り輝く」質感を設定。
	 * * 外部光源に左右されない Self-Illumination（自己発光）マップを適用することで、
	 * 暗黒の宇宙空間において常に一定の輝度を保つ恒星らしさを実現する。
	 */
	private void setupAppearance() {

		PhongMaterial material = new PhongMaterial();
		boolean textureLoaded = false;

		// 1. テクスチャのロード処理
		if (texturePath != null && !texturePath.isEmpty()) {
			try {
				Image image = new Image(getClass().getResourceAsStream(texturePath));
				if (!image.isError()) {
					// ディフューズ（拡散）だけでなく、自己発光マップとしても設定
					// これによりライトが当たらない裏側も暗くならない
					material.setDiffuseMap(image);
					material.setSelfIlluminationMap(image); 
					textureLoaded = true;
				}
			} catch (Exception e) {
				System.err.println("Sun texture load failed: " + texturePath);
			}
		}

		// 2. テクスチャ未ロード時のフォールバック（代替設定）
		if (!textureLoaded) {
			// 基本色をセット
			material.setDiffuseColor(sunColor != null ? sunColor : Color.YELLOW);
			// スペキュラー（鏡面反射）を白に設定し、光沢を出す
			material.setSpecularColor(Color.WHITE);
			// テクスチャがない場合でも、自己発光色を設定して輝きを維持
			material.setSelfIlluminationMap(null); // クリア
		}

		// 3. 形状の生成と適用
		Sphere sphere = new Sphere(radius);
		sphere.setMaterial(material);
		this.node = sphere;

	} // end setupAppearance

	/**
	 * computePosition: 太陽は公転しないが、自転による表面の動きを更新する。
	 * * 太陽は原点固定のため、x, y, z の更新は行わず、
	 * ノード自身の回転（自転）のみを毎フレーム処理する。
	 */
	@Override
	protected void computePosition() {

		// 自転角度の更新
		rotationAngle += rotationSpeed;

		// Y軸を中心に自転を適用
		if (this.node != null) {
			this.node.setRotationAxis(Rotate.Y_AXIS);
			this.node.setRotate(rotationAngle);
		}

		// 座標は常に原点を維持
		this.x = 0;
		this.y = 0;
		this.z = 0;

	} // end computePosition

} // end class SunBody
