import javafx.scene.*;
import javafx.scene.paint.PhongMaterial;
import javafx.scene.shape.Sphere;
import javafx.scene.paint.Color;

import java.util.ArrayList;

/**
 * SpaceWind: 星間物質（宇宙塵）の「流れ」をシミュレートするクラス。
 * 広大な空間にランダムなパーティクルを配置し、一定方向への移動（風）を
 * 適用することで、宇宙空間の動的な奥行きとスピード感を演出します。
 * @author きん ＆ Gemini
 */
public class SpaceWind {

	/* 星間物質（塵）を構成するノードのリスト */
	private ArrayList<Node> spaceWind = new ArrayList<>();

	// 定数定義：空間の広さとリセット境界
	private static final double RANGE = 5000.0;
	private static final double BOUNDARY = 2500.0;
	private static final double WIND_SPEED = 2.0;

	/**
	 * コンストラクタ
	 * 指定された数のパーティクルを生成し、空間内にランダムに配置します。
	 * * @param worldGroup 生成したパーティクルを追加する3D空間のグループ
	 * @param count	  配置するパーティクルの総数
	 */
	public SpaceWind(Group worldGroup, int count) {

		for (int i = 0; i < count; i++) {

			// 軽量な Sphere を塵として採用
			Sphere dust = new Sphere(1.0);

			// 1. 広い空間 (5000x5000x5000) にランダム配置
			dust.setTranslateX((Math.random() - 0.5) * RANGE);
			dust.setTranslateY((Math.random() - 0.5) * RANGE);
			dust.setTranslateZ((Math.random() - 0.5) * RANGE);

			// 2. マテリアル設定：非常に薄いグレーで、背景に溶け込む演出
			// Color.web を使用し、透過度を 0.3 に抑えることで、主張しすぎない「風」を実現。
			dust.setMaterial(new PhongMaterial(Color.web("#404040", 0.3)));

			// シーンと管理リストの両方に追加
			worldGroup.getChildren().add(dust);
			this.spaceWind.add(dust);
		}

	} // end SpaceWind constructor

	/**
	 * updateSpaceWind: 毎フレームの更新処理。
	 * 全てのパーティクルを X 軸方向に移動させ、境界を超えた場合は
	 * 反対側から再登場（ラッピング処理）させることで、無限の流れを作ります。
	 * * ※ AnimationTimer の handle メソッド内から呼び出してください。
	 */
	public void updateSpaceWind() {

		for (Node dust : this.spaceWind) {

			// X軸方向に一定速度で移動
			dust.setTranslateX(dust.getTranslateX() - WIND_SPEED);

			// 3. 境界チェックとラッピング
			// 左端 (-2500) を超えたら右端 (2500) へ戻すことで、粒子数を維持したまま流れをループさせる
			if (dust.getTranslateX() < -BOUNDARY) {
				dust.setTranslateX(BOUNDARY);
			}

		}

	} // end updateSpaceWind

} // end class SpaceWind
