import javafx.scene.*;
import javafx.scene.paint.PhongMaterial;
import javafx.scene.shape.Sphere;
import javafx.scene.paint.Color;

import java.util.Random;

/**
 * OortCloud: 太陽系の最外縁部を包み込む「オールトの雲」を生成するクラス。
 * 膨大な数の微小天体を球状に配置し、太陽系の立体的な広がりを視覚化します。
 * 描画負荷を考慮し、個々の天体はラベルを持たない軽量な Sphere として構成されます。
 * @author きん ＆ Gemini
 */
public class OortCloud {

	/**
	 * コンストラクタ
	 * 指定された数の微小天体を、太陽（原点）を中心とした球殻状にランダム配置します。
	 * @param worldGroup 生成した天体を追加する3D空間のグループ
	 * @param count	  配置する天体の総数
	 */
	public OortCloud(Group worldGroup, int count) {

		// 正規分布用
		Random random = new Random();

		for (int i = 0; i < count; i++) {

			// 1. 半径の決定
			// 正規分布（nextGaussian）を用いることで、特定の距離（2000付近）に
			// 密度を持たせた自然な塵の帯を形成します。
			double r = 2000 + random.nextGaussian() * 200;

			// 2. 球面上の座標をランダムに決定
			// 単純な乱数では極付近に密度が偏るため、Math.acos を用いて
			// 球面上に均一に分散するよう数学的に補正された分布を採用しています。
			double theta = Math.random() * 2.0 * Math.PI;	  // 方位角 (0 to 2π)
			double phi = Math.acos(2.0 * Math.random() - 1.0); // 仰角 (0 to π)

			// 3. 極座標から 3D 空間座標 (XYZ) への変換
			double x = r * Math.sin(phi) * Math.cos(theta);
			double y = r * Math.sin(phi) * Math.sin(theta);
			double z = r * Math.cos(phi);

			// 4. 軽量なパーティクル（点）として配置
			// ラベル表示や複雑なテクスチャを省き、色のみのマテリアルで描画を最適化します。
			Sphere particle = new Sphere(1.0); // 半径1.0の微小球体
			particle.setTranslateX(x);
			particle.setTranslateY(y);
			particle.setTranslateZ(z);

			// 遠方の塵を表現するため、落ち着いた DARKGREY を採用
			particle.setMaterial(new PhongMaterial(Color.DARKGREY));

			// シーンの親ノードへ追加
			worldGroup.getChildren().add(particle);

		} // end for

	} // end OortCloud constructor

} // end class OortCloud
