import javafx.scene.*;
import javafx.scene.image.Image;
import javafx.scene.shape.Sphere;
import javafx.scene.paint.PhongMaterial;
import javafx.scene.transform.Rotate;
import javafx.scene.paint.Color;
import javafx.scene.shape.CullFace;

/**
 * MilkyWayImage: 宇宙の背景として全天を包み込む「天球（スカイドーム）」を生成するクラス。
 * * 【設計のポイント】
 * 巨大な球体の内側に銀河のテクスチャを貼り付けることで、カメラがいかなる方向を向いても
 * 深淵な宇宙空間が広がる演出を行います。
 * 太陽系のスケールを遥かに凌駕する半径を設定し、視差による違和感を排除します。
 * @author きん ＆ Gemini
 */
public class MilkyWayImage {

	/**
	 * コンストラクタ
	 * * @param worldGroup 天球を追加する3D空間のメイングループ
	 */
	public MilkyWayImage(Group worldGroup) {

		// 1. 巨大な天球（スフィア）の作成
		// 太陽系の天体たちが内部に完全に収まるよう、極めて大きな半径(400,000)を設定。
		Sphere skyDome = new Sphere(400000);

		// 2. マテリアルとテクスチャの設定
		PhongMaterial mat = new PhongMaterial();

		try {
			// 銀河（天の川）の画像をロード
			// ※画像パスはプロジェクトの構成に合わせて適宜調整してください
			mat.setDiffuseMap(new Image("image/eso0932a.jpg"));
		} catch (Exception e) {
			System.err.println("MilkyWay texture load failed.");
		}

		// 3. 色合いと反射の調整
		// 背景が主張しすぎないよう、非常に低い透過度と暗めの色調に設定（引き算の美学）。
		// スペキュラー（光沢）を黒にすることで、背景がライトに反射して光るのを防ぎます。
		mat.setDiffuseColor(new Color(0.25, 0.4, 0.4, 0.01)); 
		mat.setSpecularColor(Color.BLACK);

		skyDome.setMaterial(mat);

		// 4. カリング設定（重要）
		// 通常の球体は外側を描画するが、FRONTをカリング（間引き）することで
		// 球の内側からテクスチャが見える「ドーム状」の表示を実現。
		skyDome.setCullFace(CullFace.FRONT);

		// 5. 銀河の配置（三次元回転）
		// 地球から見た実際の天の川の傾きや、中心部（射手座付近）のポジションを
		// 考慮した空間的な回転を適用します。

		// Y軸回転：中心方向の調整
		Rotate rotY = new Rotate(180, Rotate.Y_AXIS);
		// X軸回転：銀河面の傾斜
		Rotate rotX = new Rotate(15, Rotate.X_AXIS);
		// Z軸回転：全体の角度微調整
		Rotate rotZ = new Rotate(60, Rotate.Z_AXIS);

		skyDome.getTransforms().addAll(rotY, rotX, rotZ);

		// 3D空間へ配置
		worldGroup.getChildren().add(skyDome);

	} // end MilkyWayImage constructor

} // end class MilkyWayImage
